package de.stefan1200.jts3servermod.interfaces;

import java.util.HashMap;
import java.util.Vector;

/**
 * Interface of the ClientDatabaseCache class, which allows you to access the ClientDatabaseCache methods.<br><br>
 * 
 * The author of the JTS3ServerMod is not responsible for any damage or data loss!<br><br>
 * 
 * <b>E-Mail:</b><br><a href="mailto:info@stefan1200.de">info@stefan1200.de</a><br><br>
 * <b>Homepage:</b><br><a href="http://www.stefan1200.de" target="_blank">http://www.stefan1200.de</a>
 * @author Stefan Martens
 *
 */
public interface ClientDatabaseCache_Interface
{
	
	/**
	 * Returns the cached client count.
	 * @return Client count in cache or -1 if there is no local client database cache.
	 * @since 5.5
	 */
	public abstract int getClientCount();

	/**
	 * Get the last client IP address from client database id.<br>
	 * <b>Notice:</b><br>
	 * If a local cache does not exists, this search do an online search on the TS3 server.
	 * @param clientDBID Client database id
	 * @return The last IP address of the client.
	 * @since 5.0
	 */
	public abstract String getLastIP(int clientDBID);

	/**
	 * Get the client last online time from client database id.<br>
	 * <b>Notice:</b><br>
	 * If a local cache does not exists, this search do an online search on the TS3 server. The online search will return the last connected time and not the last online time!
	 * @param clientDBID Client database id
	 * @return The time stamp of the client last online time or last connected time in seconds.
	 */
	public abstract int getLastOnline(int clientDBID);

	/**
	 * Get the client created at time from client database id.<br>
	 * <b>Notice:</b><br>
	 * If a local cache does not exists, this search do an online search on the TS3 server.
	 * @param clientDBID Client database id
	 * @return The time stamp of the client creation time in seconds.
	 */
	public abstract int getCreatedAt(int clientDBID);

	/**
	 * Get the client nickname from client database id.<br>
	 * <b>Notice:</b><br>
	 * If a local cache does not exists, this search do an online search on the TS3 server.
	 * @param clientDBID Client database id
	 * @return The client nickname
	 */
	public abstract String getNickname(int clientDBID);

	/**
	 * Get the client unique id from client database id.<br>
	 * <b>Notice:</b><br>
	 * If a local cache does not exists, this search do an online search on the TS3 server.
	 * @param clientDBID Client database id
	 * @return The client unique id
	 */
	public abstract String getUniqueID(int clientDBID);

	/**
	 * Get the client description from client database id.<br>
	 * <b>Notice:</b><br>
	 * If a local cache does not exists, this search do an online search on the TS3 server.
	 * @param clientDBID Client database id
	 * @return The client description
	 */
	public abstract String getDescription(int clientDBID);

	/**
	 * Get the client database id from client unique id.<br>
	 * <b>Notice:</b><br>
	 * If a local cache does not exists, this search do an online search on the TS3 server.
	 * @param clientUniqueID Client Unique ID
	 * @return The client database id
	 * @since 5.3
	 */
	public abstract int getDatabaseID(String clientUniqueID);
	
	/**
	 * Check if a local cache of the TS3 client database exists. 
	 * @return Returns <code>true</code>, if a local cache of the TS3 client database exists. 
	 * @since 5.5
	 */
	public abstract boolean isCacheLocal();

	/**
	 * Check if the local client database cache is still updating.
	 * @return Returns <code>true</code>, if the cache is still updating. 
	 */
	public abstract boolean isUpdateRunning();

	/**
	 * Search for clients who was not online in the last X days.<br>
	 * <b>Notice:</b><br>
	 * A local client database cache is required!
	 * @param daysInactive Days of inactivity
	 * @param sortOrder Set to <code>1</code> to get the list sorted from oldest to youngest last online time, <code>0</code> for youngest to oldest sorting, <code>-1</code> to disable sorting.
	 * @return Returns a Vector with HashMaps containing the client database ids (cldbid) and last online time (lastonline).
	 * @since 3.5.1
	 */
	public abstract Vector<HashMap<String, Integer>> searchInactiveClients(int daysInactive, int sortOrder);

	/**
	 * Search for clients with a specified IP address. Star wildcard character is possible, example: 127.0.*<br>
	 * <b>Notice:</b><br>
	 * A local client database cache is required!
	 * @param search The search string with the ip address (or parts of it)
	 * @return Returns a Vector with client database ids.
	 * @since 5.0
	 */
	public abstract Vector<Integer> searchIPAddress(String search);

	/**
	 * Search for clients with a specified description. Star wildcard character is possible, example: foo*bar<br>
	 * <b>Notice:</b><br>
	 * A local client database cache is required!
	 * @param search The search string with the description (or parts of it)
	 * @return Returns a Vector with client database ids.
	 * @since 6.2
	 */
	public abstract Vector<Integer> searchDescription(String search);

	/**
	 * Search for clients with a specified client name or unique id. Star wildcard character is possible at the client name, example: foo*bar<br>
	 * <b>Notice:</b><br>
	 * If a local cache does not exists, this search do an online search on the TS3 server. 
	 * @param search The search string with the client name (or parts of it) or the unique id.
	 * @return Returns a Vector with client database ids.
	 */
	public abstract Vector<Integer> searchClientNickname(String search);
	
	/**
	 * Delete a client from the client database cache. This has no effect to the Teamspeak 3 database.
	 * @param clientDBID Client database id
	 * @return Returns <code>true</code>, if the client was found and removed from the local Client Database Cache, <code>false</code> if not.
	 * @since 6.3
	 */
	public abstract boolean deleteClientFromCache(int clientDBID);

}