package de.stefan1200.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.PrintStream;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.sql.Types;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Properties;
import java.util.Vector;

import de.stefan1200.jts3servermod.interfaces.JTS3ServerMod_Interface;
/**
 * ArrangedPropertiesWriter allows you to write a formatted property file with help texts for every key.
 * Since version 2.0 it is possible to use a MySQL database as data source.
 * <br><br>
 * You are allowed to use this file for free, but it would be nice to be named in the credits of your project.
 * The author of this class is not responsible for any damage or data loss!
 * It is not allowed to sell this class for money, it has to be free to get!
 * <br><br>
 * <b>E-Mail:</b><br><a href="mailto:info@stefan1200.de">info@stefan1200.de</a><br><br>
 * <b>Homepage:</b><br><a href="http://www.stefan1200.de" target="_blank">http://www.stefan1200.de</a>
 * 
 * @author Stefan Martens
 * @version 2.4 (07.10.2016)
 */
public class ArrangedPropertiesWriter extends Object implements Cloneable
{
	private final String SEPARATOR = "***";
	private final String LINE_SEPARATOR = System.getProperty("line.separator");
	private HashMap<String, String> hmHelp = new HashMap<String, String>();
	private HashMap<String, String> hmValue = new HashMap<String, String>();
	private HashMap<String, Boolean> hmSave = new HashMap<String, Boolean>();
	private Vector<String> vKeys = new Vector<String>();
	private SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	private JTS3ServerMod_Interface mainClass = null;
	private boolean writeProtected = false;
	
	private MySQLConnect lastMysqlConnect = null;
	private int lastMysqlInstanceID = -1;
	private String lastConfigFileName = null;
	private String lastConfigFileHeader = null;
	
	/**
	 * Create a new instance of the ArrangedPropertiesWriter.
	 * @param mainClass A new main class, will be needed to access the addLogEntry method to write log entries.
	 */
	public ArrangedPropertiesWriter(JTS3ServerMod_Interface mainClass)
	{
		this.mainClass = mainClass;
	}
	
	/**
	 * Create a new instance with an already loaded configuration.
	 * @param mainClass A new main class, will be needed to access the addLogEntry method to write log entries.
	 * @param hmHelp Instance of hmHelp
	 * @param hmValue Instance of hmValue
	 * @param hmSave Instance of hmSave
	 * @param vKeys Instance of vKeys
	 */
	public ArrangedPropertiesWriter(JTS3ServerMod_Interface mainClass, HashMap<String, String> hmHelp, HashMap<String, String> hmValue, HashMap<String, Boolean> hmSave, Vector<String> vKeys)
	{
		this.mainClass = mainClass;
		this.hmHelp = hmHelp;
		this.hmValue = hmValue;
		this.hmSave = hmSave;
		this.vKeys = vKeys;
	}
	
	/**
	 * Get a new instance of the ArrangedPropertiesWriter with the same loaded configuration. The main class will be set to <code>null</code>.<br><br>
	 * <b>Important:</b><br>
	 * This is not a deep copy, if the loaded configuration will be changed at the original ArrangedPropertiesWriter instance, this copy will change too.
	 * @return Returns a shallow copy of this ArrangedPropertiesWriter.
	 */
	public ArrangedPropertiesWriter clone()
	{
		return new ArrangedPropertiesWriter(null, this.hmHelp, this.hmValue, this.hmSave, this.vKeys);
	}
	
	/**
	 * Set a new main class. Will be needed to access the addLogEntry method to write log entries.
	 * @param mainClass A new main class, will be needed to access the addLogEntry method to write log entries.
	 */
	public void setNewMainClass(JTS3ServerMod_Interface mainClass)
	{
		this.mainClass = mainClass;
	}
	
	/**
	 * Set write protection to <code>true</code> to make sure, that no values will be changed, removed or added.
	 * @param flag Set <code>true</code> to prevent any changes to the configuration.
	 * @since 2.2
	 */
	public void setWriteProtection(boolean flag)
	{
		this.writeProtected = flag;
	}
	
	/**
	 * Set a default config file header. Useless for MySQL.
	 * @param header A simple config file header as String.
	 * @since 2.4
	 */
	public void setDefaultConfigFileHeader(String header)
	{
		this.lastConfigFileHeader = header;
	}
	
	/**
	 * Add a new key at the current position.
	 * @param key Name of the key
	 * @param helpText Help text of the key or <code>null</code> if not wanted.
	 * @return <code>true</code> if the key was added, <code>false</code> if the key is already known.
	 */
	public boolean addKey(String key, String helpText)
	{
		return addKey(key, helpText, null, true);
	}
	
	/**
	 * Add a new key at the current position.
	 * @param key Name of the key
	 * @param helpText Help text of the key or <code>null</code> if not wanted.
	 * @param defaultValue Preset the key with a default value. Set <code>null</code> if no default value should be set!
	 * @return <code>true</code> if the key was added, <code>false</code> if the key is already known.
	 */
	public boolean addKey(String key, String helpText, String defaultValue)
	{
		return addKey(key, helpText, defaultValue, true);
	}
	
	/**
	 * Add a new key at the current position.
	 * @param key Name of the key
	 * @param helpText Help text of the key or <code>null</code> if not wanted.
	 * @param saveToFile False if this config key should be ignored by the save method, if saving to file.
	 * @return <code>true</code> if the key was added, <code>false</code> if the key is already known.
	 */
	public boolean addKey(String key, String helpText, boolean saveToFile)
	{
		return addKey(key, helpText, null, saveToFile);
	}
	
	/**
	 * Add a new key at the current position.
	 * @param key Name of the key
	 * @param helpText Help text of the key or <code>null</code> if not wanted.
	 * @param defaultValue Preset the key with a default value. Set <code>null</code> if no default value should be set!
	 * @param saveToFile False if this config key should be ignored by the save method, if saving to file.
	 * @return <code>true</code> if the key was added, <code>false</code> if the key is already known.
	 */
	public boolean addKey(String key, String helpText, String defaultValue, boolean saveToFile)
	{
		if (writeProtected) return false;
		
		if(!key.equals(SEPARATOR) && key.length() > 0 && vKeys.indexOf(key) == -1)
		{
			vKeys.addElement(key);
			hmValue.put(key, defaultValue);
			hmHelp.put(key, helpText);
			hmSave.put(key, saveToFile);
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Inserts a new key at the given position.
	 * @param key Name of the key
	 * @param pos Insert position
	 * @param helpText Help text of the key or <code>null</code> if not wanted.
	 * @return <code>true</code> if the key was inserted, <code>false</code> if the key is already known.
	 */
	public boolean insertKey(String key, int pos, String helpText)
	{
		if (writeProtected) return false;
		
		if(!key.equals(SEPARATOR) && key.length() > 0 && vKeys.indexOf(key) == -1)
		{
			vKeys.insertElementAt(key, pos);
			hmHelp.put(key, helpText);
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Returns if this value can save to file.
	 * @param key Name of the key
	 * @return <code>true</code> if the key can be saved to file or <code>false</code> if not.
	 */
	public boolean canSaveToFile(String key)
	{
		return hmSave.get(key);
	}
	
	/**
	 * Get a list of all known keys.
	 * @return List of keys
	 */
	public Vector<String> getKeys()
	{
		Vector<String> retKeys = new Vector<String>();
		
		retKeys.addAll(vKeys);
		while (retKeys.removeElement(SEPARATOR));
		
		return retKeys;
	}
	
	/**
	 * Returns the value of the key.
	 * @param key Name of the key
	 * @return The value of the key or <code>null</code> if there is no value.
	 */
	public String getValue(String key)
	{
		return hmValue.get(key);
	}
	
	/**
	 * Returns the value of the key. If there is no value, the defValue will be returned.
	 * @param key Name of the key
	 * @param defValue Default value
	 * @return The value of the key or the default value if there is no value.
	 */
	public String getValue(String key, String defValue)
	{
		if (hmValue.get(key) == null)
		{
			return defValue;
		}
		else
		{
			return hmValue.get(key);
		}
	}
	
	/**
	 * Sets a value for the given key.
	 * @param key Name of the key
	 * @param value Value for the key
	 * @return <code>true</code> if the value was set, <code>false</code> if key not found.
	 */
	public boolean setValue(String key, String value)
	{
		if (writeProtected) return false;
		
		if(!key.equals(SEPARATOR) && key.length() > 0 && vKeys.indexOf(key) != -1)
		{
			hmValue.put(key, value);
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Sets a value for the given key.
	 * @param key Name of the key
	 * @param value Value for the key, will be converted to String.
	 * @return <code>true</code> if the value was set, <code>false</code> if key not found.
	 */
	public boolean setValue(String key, long value)
	{
		return setValue(key, Long.toString(value));
	}

	/**
	 * Sets a value for the given key.
	 * @param key Name of the key
	 * @param value Value for the key, will be converted to String.
	 * @return <code>true</code> if the value was set, <code>false</code> if key not found.
	 */
	public boolean setValue(String key, double value)
	{
		return setValue(key, Double.toString(value));
	}
	
	/**
	 * Sets a value for the given key.
	 * @param key Name of the key
	 * @param value Value for the key, will be converted to String.
	 * @return <code>true</code> if the value was set, <code>false</code> if key not found.
	 */
	public boolean setValue(String key, boolean value)
	{
		return setValue(key, Boolean.toString(value));
	}
	
	/**
	 * Clears all values.
	 */
	public void removeAllValues()
	{
		hmValue.clear();
	}
	
	/**
	 * Add a separator at the current position.
	 */
	public void addSeparator()
	{
		if (writeProtected) return;
		
		vKeys.addElement(SEPARATOR);
	}
	
	/**
	 * Insert separator at a given position.
	 * @param pos Position to insert separator.
	 */
	public void insertSeparator(int pos)
	{
		if (writeProtected) return;
		
		vKeys.insertElementAt(SEPARATOR, pos);
	}
	
	/**
	 * Removes all separators.
	 */
	public void removeAllSeparators()
	{
		if (writeProtected) return;
		
		while (vKeys.removeElement(SEPARATOR));
	}
	
	/**
	 * Returns number of stored keys.
	 * @return Key count
	 */
	public int getKeyCount()
	{
		return vKeys.size();
	}
	
	/**
	 * Get the help text of a key.
	 * @param key Name of the key
	 * @return The help text as String or <code>null</code> if the key has no help text.
	 */
	public String getHelpText(String key)
	{
		return hmHelp.get(key);
	}
	
	/**
	 * Remove a key. The help text and value of the key will be also removed.
	 * @param key Name of the key
	 * @return <code>true</code> if the key was removed, <code>false</code> if key not found.
	 */
	public boolean removeKey(String key)
	{
		if (writeProtected) return false;
		
		if(!key.equals(SEPARATOR) && key.length() > 0 && vKeys.indexOf(key) != -1)
		{
			vKeys.removeElement(key);
			hmHelp.remove(key);
			hmValue.remove(key);
			hmSave.remove(key);
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Load values from property file. Only the values of known keys will be read from file.
	 * @param file File object of the property file.
	 * @return <code>true</code> if file was successfully read, <code>false</code> if not.
	 */
	public boolean loadValues(File file)
	{
		if (writeProtected) return false;
		
		if (file == null || !file.isFile())
		{
			return false;
		}
		
		lastConfigFileName = file.getAbsolutePath();
		lastMysqlConnect = null;
		lastMysqlInstanceID = -1;
		
		Properties prop = new Properties();
		
		try
		{
			prop.load(new FileInputStream(file));
			String temp;
			
			for (String key : vKeys)
			{
				temp = prop.getProperty(key);
				if (temp != null)
				{
					hmValue.put(key, temp);
				}
			}
		}
		catch (Exception e)
		{
			if (mainClass != null) mainClass.addLogEntry(null, e, false);
			prop = null;
			return false;
		}
		
		prop = null;
		return true;
	}
	
	/**
	 * Load values from property file. Only the values of known keys will be read from file.
	 * @param filename Path to the property file.
	 * @return <code>true</code> if file was successfully read, <code>false</code> if not.
	 */
	public boolean loadValues(String filename)
	{
		if (filename == null)
		{
			return false;
		}
		
		return loadValues(new File(filename));
	}
	
	/**
	 * Load values from SQL database. Only the values of known keys will be read from database.
	 * @param mysqlConnect <code>MySQLConnect</code> class.
	 * @param instanceID The instance ID in the database.
	 * @return <code>true</code> if database was successfully read, <code>false</code> if not.
	 */
	public boolean loadValues(MySQLConnect mysqlConnect, int instanceID)
	{
		if (writeProtected) return false;
		
		lastMysqlConnect = mysqlConnect;
		lastMysqlInstanceID = instanceID;
		lastConfigFileName = null;
		
		if (mysqlConnect == null || instanceID < 1)
		{
			return false;
		}
		
		boolean retValue = false;
		Statement st = null;
		ResultSet rs = null;
		
		try
		{
			mysqlConnect.connect();
			st = mysqlConnect.getStatement();
			rs = st.executeQuery("SELECT configkey, configvalue FROM jts3servermod_instanceconfig WHERE instance_id = " + Integer.toString(instanceID));
			String temp;
			String keyDB;
			
			while(rs.next())
			{
				keyDB = rs.getString(1);
				for (String key : vKeys)
				{
					if (key.equals(keyDB))
					{
						temp = rs.getString(2);
						if (temp != null)
						{
							hmValue.put(key, temp);
						}
						break;
					}
				}
			}
			
			retValue = true;
		}
		catch (Exception e)
		{
			if (mainClass != null) mainClass.addLogEntry(null, e, false);
			retValue = false;
		}
		finally
		{
			try {if (rs != null) rs.close();} catch (Exception e2) {}
			try {if (st != null) st.close();} catch (Exception e3) {}
			mysqlConnect.close();
		}
		
		return retValue;
	}
	
	/**
	 * Save values to the last loading location! 
	 * @return <code>true</code> if values was successfully written, <code>false</code> if not.
	 * @since 2.4
	 */
	public boolean save()
	{
		if (lastMysqlConnect != null && lastMysqlInstanceID >= 0)
			return save(lastMysqlConnect, lastMysqlInstanceID);
		else if (lastConfigFileName != null)
			return save(lastConfigFileName, lastConfigFileHeader);
		else
			return false;
	}
	
	/**
	 * Save values to SQL database.
	 * @param mysqlConnect <code>MySQLConnect</code> class.
	 * @param instanceID The instance ID in the database.
	 * @return <code>true</code> if values was successfully written to database, <code>false</code> if not.
	 */
	public boolean save(MySQLConnect mysqlConnect, int instanceID)
	{
		if (mysqlConnect == null || instanceID < 1)
		{
			return false;
		}
		
		boolean retValue = false;
		Statement st = null;
		PreparedStatement pst = null;
		
		try
		{
			mysqlConnect.connect();
			st = mysqlConnect.getStatement();
			st.executeUpdate("DELETE FROM jts3servermod_instanceconfig WHERE instance_id = " + Integer.toString(instanceID));
			st.close();
			st = null;
			
			pst = mysqlConnect.getPreparedStatement("INSERT INTO jts3servermod_instanceconfig (instance_id, configkey, configvalue) VALUES (" + Integer.toString(instanceID) + ", ?, ?)");
			
			for (String key : vKeys)
			{
				if (key.equals(SEPARATOR))
				{
					continue;
				}
				
				pst.setString(1, key);
				if (hmValue.get(key) != null)
				{
					pst.setString(2, hmValue.get(key));
				}
				else
				{
					pst.setNull(2, Types.VARCHAR);
				}
				
				pst.executeUpdate();
			}
			
			retValue = true;
		}
		catch (Exception e)
		{
			if (mainClass != null) mainClass.addLogEntry(null, e, false);
			retValue = false;
		}
		finally
		{
			try {if (pst != null) pst.close();} catch (Exception e2) {}
			try {if (st != null) st.close();} catch (Exception e3) {}
			mysqlConnect.close();
		}
		
		return retValue;
	}
	
	/**
	 * Write arranged property file to disk.
	 * @param filename Path where the file should be saved.
	 * @param header The header of the file or <code>null</code> if not wanted.
	 * @return <code>true</code> if file was successfully written, <code>false</code> if not.
	 */
	public boolean save(String filename, String header)
	{
		if (filename == null)
		{
			return false;
		}
		
		PrintStream ps;
		try
		{
			ps = new PrintStream(filename, "ISO-8859-1");
		}
		catch (Exception e)
		{
			if (mainClass != null) mainClass.addLogEntry(null, e, false);
			return false;
		}
		
		if (header != null && header.length() > 0)
		{
			ps.println(convertString(header));
		}
		
		ps.println("# File created at " + sdf.format(new Date(System.currentTimeMillis())));
		ps.println();
		
		for (String key : vKeys)
		{
			if (key.equals(SEPARATOR))
			{
				ps.println();
				continue;
			}
			
			if (hmSave.get(key))
			{
				if (hmHelp.get(key) != null)
				{
					ps.println(convertString(hmHelp.get(key)));
				}
				
				ps.print(key);
				ps.print(" = ");
				ps.println((hmValue.get(key) == null ? "" : hmValue.get(key)));
			}
		}
		
		ps.close();
		
		return true;
	}
	
	private String convertString(String text)
	{
		String retValue = "# " + text;
		
		retValue = retValue.replace("\\", "$[mkbackslashsave]");
		retValue = retValue.replace("\n", LINE_SEPARATOR + "# ");
		retValue = retValue.replace("$[mkbackslashsave]", "\\");
		
		return retValue;
	}
}
