package de.stefan1200.jts3servermod.interfaces;

import java.util.HashMap;
import java.util.TimerTask;
import java.util.Vector;

import de.stefan1200.util.MySQLConnect;

/**
 * Interface of the JTS3ServerMod class, which allows you to use the public JTS3ServerMod methods to make your plugin development easier.<br><br>
 * 
 * The author of the JTS3ServerMod is not responsible for any damage or data loss!<br><br>
 * 
 * <b>E-Mail:</b><br><a href="mailto:info@stefan1200.de">info@stefan1200.de</a><br><br>
 * <b>Homepage:</b><br><a href="http://www.stefan1200.de" target="_blank">http://www.stefan1200.de</a>
 * @author Stefan Martens
 *
 */
public interface JTS3ServerMod_Interface
{
	/**
	 * Human readable version string, which will be used everywhere to show JTS3ServerMod version to users.
	 */
	public static final String VERSION = "6.3.1 (20.11.2016)";
	
	/**
	 * Internal version number for comparing version for version check or plugins.
	 * @since 5.3 
	 */
	public static final long VERSION_BUILD = 6301;
	
	/**
	 * Use this log level for debugging purposes only.
	 * @since 5.3
	 */
	public final static byte ERROR_LEVEL_DEBUG = 0;
	
	/**
	 * Use this log level for info messages (something was done successful or stuff like that).
	 * @since 5.3
	 */
	public final static byte ERROR_LEVEL_INFO = 1;
	
	/**
	 * Use this log level to warn the user about a possible problem, but everything could work like expected. 
	 * @since 5.3
	 */
	public final static byte ERROR_LEVEL_WARNING = 2;
	
	/**
	 * Use this log level if an error occurred or the configuration is wrong.
	 * In most cases the function will be disabled if an error occur,
	 * but the bot with all other functions is working without further problems.
	 * @since 5.3
	 */
	public final static byte ERROR_LEVEL_ERROR = 3;
	
	/**
	 * Use this log level if a critical error occurred, which have to stop the bot in most cases.
	 * In normal cases no function trigger critical errors, only the bot itself.
	 * @since 5.3
	 */
	public final static byte ERROR_LEVEL_CRITICAL = 4;
	
	/**
	 * Use this constant at the setListModes() method. Adds <code>client_away</code> and <code>client_away_message</code> to the clientlist.
	 * @see LoadConfiguration#setListModes(java.util.BitSet)
	 */
	public final static int LIST_AWAY = 0;
	
	/**
	 * Use this constant at the setListModes() method. Adds <code>client_servergroups</code>, <code>client_channel_group_id</code> and <code>client_channel_group_inherited_channel_id</code> to the clientlist.
	 * @see LoadConfiguration#setListModes(java.util.BitSet)
	 */
	public final static int LIST_GROUPS = 1;
	
	/**
	 * Use this constant at the setListModes() method. Adds <code>client_version</code> and <code>client_platform</code> to the clientlist.
	 * @see LoadConfiguration#setListModes(java.util.BitSet)
	 */
	public final static int LIST_INFO = 2;
	
	/**
	 * Use this constant at the setListModes() method. Adds <code>client_idle_time</code>, <code>client_created</code> and <code>client_lastconnected</code> to the clientlist.
	 * @see LoadConfiguration#setListModes(java.util.BitSet)
	 */
	public final static int LIST_TIMES = 3;
	
	/**
	 * Use this constant at the setListModes() method. Adds <code>client_unique_identifier</code> to the clientlist.
	 * @see LoadConfiguration#setListModes(java.util.BitSet)
	 */
	public final static int LIST_UID = 4;
	
	/**
	 * Use this constant at the setListModes() method. Adds <code>client_flag_talking</code>, <code>client_input_muted</code>, <code>client_output_muted</code>, <code>client_input_hardware</code>, <code>client_output_hardware</code>, <code>client_talk_power</code>, <code>client_is_talker</code>, <code>client_is_priority_speaker</code>, <code>client_is_recording</code> and <code>client_is_channel_commander</code> to the clientlist.
	 * @see LoadConfiguration#setListModes(java.util.BitSet)
	 */
	public final static int LIST_VOICE = 5;
	
	/**
	 * Use this constant at the setListModes() method. Adds <code>client_country</code> to the clientlist.
	 * @see LoadConfiguration#setListModes(java.util.BitSet)
	 */
	public final static int LIST_COUNTRY = 6;
	
	/**
	 * Use this constant at the setListModes() method. Adds <code>connection_client_ip</code> to the clientlist.
	 * @see LoadConfiguration#setListModes(java.util.BitSet)
	 */
	public final static int LIST_IP = 7;

	/**
	 * Get the UTF-8 byte size of a string with UTF-8 characters, important for TS3 chat messages (with a maximum of 1024).
	 * @param sequence The string
	 * @return The UTF-8 byte size of the string
	 */
	public int getUTF8Length(CharSequence sequence);
	
	/**
	 * Returns the message encoding, which was selected by the user.
	 * @return The message encoding, default is UTF-8
	 */
	public abstract String getMessageEncoding();

	/**
	 * Returns the channel name of the channel id. This will search in the channel list cache, which will be updated every minute.
	 * @param channelID The channel id of the channel
	 * @return The name of the channel or <code>null</code>, if the channel id was not found (or is currently not cached).
	 */
	public abstract String getChannelName(int channelID);

	/**
	 * Use this to unload a function. Normally used to unload the own object, so you can simple use <code>unloadFunction(this)</code> within the own class.
	 * @param c The class of the function.
	 */
	public abstract void unloadFunction(Object c);

	/**
	 * Loads messages from a simple messages file by keeping the messages encoding set by the user. Like at the original messages files from the JTS3ServerMod, the first line have to be:<br>
	 * <code># JTS3ServerMod Config File</code><br>
	 * All lines starting with # or lines with less than 4 characters will be skipped. If there are more lines, than the array size of <code>configValues</code>, all following lines will be ignores.<br>
	 * <br>
	 * Example for loading two messages from a file:<br>
	 * <pre> String[] configKeys = {configPrefix + "_firstMessage", configPrefix + "_secondMessage"};
	 * if (!modClass.loadMessages(configPrefix, "_file", configKeys))
	 * {
	 * 	throw new BotConfigurationException("Messages could not be loaded!");
	 * }
	 * String firstMessage = config.getValue(configKeys[0]);
	 * String secondMessage = config.getValue(configKeys[1]);</pre>
	 * @param configPrefix The config prefix from {@link HandleBotEvents#initClass(JTS3ServerMod_Interface, de.stefan1200.jts3serverquery.JTS3ServerQuery, String)}
	 * @param configKey_Path The config suffix which contains the file path to the messages.
	 * @param configValues A String array with the config key names, where the messages should be stored. Needs to be in the same order as in the messages file.
	 * @return Returns <code>false</code>, if anything was going wrong while loading the messages. More information are printed into the log file. <code>true</code> if everything was good or the MySQL mode is activated (in this case all messages are already loaded from the database).
	 */
	public abstract boolean loadMessages(String configPrefix, String configKey_Path, String[] configValues);

	/**
	 * Returns the used instance of the ServerInfoCache, which contains some information about the currently connected server.
	 * @return The used instance of the ServerInfoCache
	 * @see ServerInfoCache_Interface
	 */
	public abstract ServerInfoCache_Interface getServerInfoCache();

	/**
	 * Returns the last loaded client list from the currently connected Teamspeak 3 server.<br>Will be loaded every check interval (can be changed by the user, default is every second).
	 * @return The last loaded client list
	 */
	public abstract Vector<HashMap<String, String>> getClientList();

	/**
	 * Returns the last loaded channel list from the currently connected Teamspeak 3 server.<br>Will be updated every minute.
	 * @return The last loaded channel list
	 */
	public abstract Vector<HashMap<String, String>> getChannelList();

	/**
	 * Returns the used instance of the MySQL connection. You can use this, if you have to access the database of the JTS3ServerMod. In most cases you don't need this.
	 * @return The used instance of the MySQL connection.
	 * @see MySQLConnect
	 */
	public abstract MySQLConnect getMySQLConnection();

	/**
	 * Returns the virtual bot instance id in the MySQL database. Only needed if you access the MySQL database on your own. In most cases you don't need this.
	 * @return The virtual bot instance id
	 */
	public abstract int getInstanceID();

	/**
	 * Returns the default channel id of the Teamspeak 3 server. Will be updated every minute.
	 * @return The default channel id
	 */
	public abstract int getDefaultChannelID();

	/**
	 * Returns the used instance of the ClientDatabaseCache, which contains the client database cache, like unique id, ip address, names and last connected time of all clients, of the currently connected server.
	 * @return The used instance of the ClientDatabaseCache
	 * @see ClientDatabaseCache_Interface
	 */
	public abstract ClientDatabaseCache_Interface getClientCache();

	/**
	 * Returns the current check interval of the JTS3ServerMod. Can be changed by the user, default is every second.
	 * @return The current check interval
	 */
	public abstract int getCheckInterval();

	/**
	 * Converts the time stamp in milliseconds to a string containing the date and time with the user defined formatting rules. 
	 * @param timestamp The time stamp you want to convert into a String. 
	 * @return A string containing the date and time with the user defined formatting rules.
	 */
	public abstract String getStringFromTimestamp(long timestamp);

	/**
	 * Register function for Teamspeak 3 channel events. Events will call the method {@link HandleTS3Events#handleClientEvents(String, HashMap)} at the given class.
	 * @param obj The class of the function which implements the {@link HandleTS3Events} interface.
	 * @see HandleTS3Events#handleClientEvents(String, HashMap)
	 */
	public abstract void addTS3ChannelEvent(Object obj);

	/**
	 * Register function for Teamspeak 3 server events. Events will call the method {@link HandleTS3Events#handleClientEvents(String, HashMap)} at the given class.
	 * @param obj The class of the function which implements the {@link HandleTS3Events} interface.
	 * @see HandleTS3Events#handleClientEvents(String, HashMap)
	 */
	public abstract void addTS3ServerEvent(Object obj);

	/**
	 * Add your TimerTask to the bot timer.<br>
	 * <br>
	 * <b>Important:</b><br>
	 * Never create an own Timer in your class. Only create a TimerTask and add it to the bot timer. This make sure that the bot can stop all used timers at reloading or stopping the bot. 
	 * @param task Your TimerTask
	 * @param firstStart The first start time in milliseconds
	 * @param interval The interval of the TimerTask in milliseconds
	 */
	public abstract void addBotTimer(TimerTask task, long firstStart, long interval);

	/**
	 * Returns if the global message variables are enabled or disabled. If enabled, you can use the {@link JTS3ServerMod_Interface#replaceGlobalMessageVars(String)} method to replace the global message variables.<br>
	 * This is not needed, if you use the {@link JTS3ServerMod_Interface#sendMessageToClient(String, String, int, String)} method.
	 * @return Returns <code>false</code>, if the global message variables are disabled, <code>true</code> if enabled.
	 * @see JTS3ServerMod_Interface#replaceGlobalMessageVars(String)
	 */
	public abstract boolean isGlobalMessageVarsEnabled();

	/**
	 * Replace the global message variables at the given string. You should check first, if the user enabled this feature.<br>
	 * This is not needed, if you use the {@link JTS3ServerMod_Interface#sendMessageToClient(String, String, int, String)} method.
	 * @param message The message which might contain global message variables.
	 * @return The message with the global message variables replaced
	 * @see JTS3ServerMod_Interface#isGlobalMessageVarsEnabled()
	 */
	public abstract String replaceGlobalMessageVars(String message);

	/**
	 * Send a chat or poke message to a client. If enabled, global message variables will be replaced automatically. Also the message split sequence <code>+</code> will be parsed automatically and will send multiple messages to the client.
	 * @param configPrefix The config prefix from {@link HandleBotEvents#initClass(JTS3ServerMod_Interface, de.stefan1200.jts3serverquery.JTS3ServerQuery, String)}
	 * @param mode <code>chat</code> or <code>poke</code> is possible here.
	 * @param clientID The client id of the target client.
	 * @param message The message as string.
	 * @return Returns <code>false</code>, if anything was going wrong while sending the messages to the client. More information are printed into the log file. <code>true</code> if everything was good.
	 */
	public abstract boolean sendMessageToClient(String configPrefix, String mode, int clientID, String message);

	/**
	 * Request the maximum allowed length for messages. Chat message length is counted in byte, poke and kick messages in character count.<br>
	 * <br>
	 * <b>Notice:</b><br>
	 * You should use {@link JTS3ServerMod_Interface#isMessageLengthValid(String, String)} instead.
	 * @param type <code>chat</code>, <code>kick</code> or <code>poke</code> is possible here.
	 * @return Returns the allowed length or bytes.
	 * @see JTS3ServerMod_Interface#isMessageLengthValid(String, String)
	 */
	public abstract short getMaxMessageLength(String type);

	/**
	 * Check if the message length is valid.
	 * @param type <code>chat</code>, <code>kick</code> or <code>poke</code> is possible here.
	 * @param message The message as string.
	 * @return Returns <code>false</code>, if the message length is not valid. <code>true</code> if everything is good.
	 */
	public abstract boolean isMessageLengthValid(String type, String message);

	/**
	 * Extracts the client_idle_time from the clientInfo HashMap, converts this to a long and writes down a warning to the bot log, if the Teamspeak 3 server returned an invalid value (which can happen rarely).
	 * @param clientInfo HashMap containing the client info
	 * @param ignoreInChannel Set a channel id here, to prevent writing a warning to the bot log, if the client is in the given channel id.
	 * @return The client idle time
	 */
	public abstract long getIdleTime(HashMap<String, String> clientInfo, int ignoreInChannel);

	/**
	 * Check if a Vector list contains a given id.
	 * @param searchID The id
	 * @param list The list as Vector containing Integer
	 * @return Returns <code>true</code>, if the list contains the given id, <code>false</code> if not.
	 */
	public abstract boolean isIDListed(int searchID, Vector<Integer> list);

	/**
	 * Returns the server group list cache. Will be updated every minute.
	 * @return The server group list cache as HashMap in a Vector.
	 * @since 6.2.0
	 */
	public abstract Vector<HashMap<String, String>> getServerGroupCache();
	
	/**
	 * Returns the server group name of the server group id. This will search in the server group cache, which will be updated every minute.
	 * @param groupID The server group id of the server group
	 * @return The name of the server group or <code>null</code>, if the server group id was not found (or is currently not cached).
	 */
	public abstract String getServerGroupName(int groupID);

	/**
	 * Returns the server group type of the server group id. This will search in the server group cache, which will be updated every minute.
	 * @param groupID The server group id of the server group
	 * @return The type of the server group or -1, if the server group id was not found (or is currently not cached).
	 */
	public abstract int getServerGroupType(int groupID);
	
	/**
	 * Returns the channel group list cache. Will be updated every minute.
	 * @return The channel group list cache as HashMap in a Vector.
	 * @since 6.2.0
	 */
	public abstract Vector<HashMap<String, String>> getChannelGroupCache();
	
	/**
	 * Returns the channel group name of the channel group id. This will search in the channel group cache, which will be updated every minute.
	 * @param groupID The channel group id of the channel group
	 * @return The name of the channel group or <code>null</code>, if the channel group id was not found (or is currently not cached).
	 * @since 6.2.0
	 */
	public abstract String getChannelGroupName(int groupID);

	/**
	 * Returns the channel group type of the channel group id. This will search in the channel group cache, which will be updated every minute.
	 * @param groupID The channel group id of the channel group
	 * @return The type of the channel group or -1, if the channel group id was not found (or is currently not cached).
	 * @since 6.2.0
	 */
	public abstract int getChannelGroupType(int groupID);

	/**
	 * Returns the first group id, which is listed in the Vector list and in the comma separated list of groups as well.
	 * @param groupIDs A comma separated list of groups
	 * @param list The list as Vector containing Integer
	 * @return The first group id which is in both lists.
	 */
	public abstract int getListedGroup(String groupIDs, Vector<Integer> list);

	/**
	 * Check if the Vector list contains any of the comma separated list of groups.
	 * @param groupIDs A comma separated list of groups
	 * @param list The list as Vector containing Integers
	 * @return Returns <code>true</code>, if the list contains the given id, <code>false</code> if not.
	 */
	public abstract boolean isGroupListed(String groupIDs, Vector<Integer> list);
	
	/**
	 * Check if the comma separated list of groups contains the given group id.
	 * @param groupIDs A comma separated list of groups
	 * @param searchGroupID A single group id as int
	 * @return Returns <code>true</code>, if the list contains the given id, <code>false</code> if not.
	 * @since 6.1.0
	 */
	public abstract boolean isGroupListed(String groupIDs, int searchGroupID);

	/**
	 * Check if the given unique id is in the Vector.
	 * @param uidList The list as Vector containing Strings
	 * @param uid The unique id as String
	 * @return Returns <code>true</code>, if the list contains the given unique id, <code>false</code> if not.
	 * @since 6.1.0
	 */
	public abstract boolean isUIDListed(Vector<String> uidList, String uid);

	/**
	 * Reformats the client version string of the Teamspeak 3 client into a human readable version string with date and time.
	 * @param version The Teamspeak 3 client version string from the clientlist or clientinfo.
	 * @return A human readable version string with date and time.
	 */
	public abstract String getVersionString(String version);

	/**
	 * Reformats the a size in bytes into human readable GB/GiB, MB/MiB or KB/KiB sizes, depends on the size.
	 * @param size The size in bytes as long.
	 * @param base1000 Set <code>true</code> to use the base 1000 for the size.
	 * @return The size as String.
	 */
	public abstract String getFileSizeString(long size, boolean base1000);

	/**
	 * Returns a human readable String with the difference between two time stamps.<br>
	 * The result could be something like:<br>
	 * 2 days and 10:28:12 hours<br>
	 * 1:28 minutes<br>
	 * 42 seconds
	 * @param from Smaller time stamp
	 * @param to Higher time stamp
	 * @return A human readable String with the difference between two time stamps.
	 */
	public abstract String getDifferenceTime(long from, long to);

	/**
	 * Add a log entry to the bot log. Possible log levels can be found in the {@link JTS3ServerMod_Interface}, constant names starts with ERROR_LEVEL_.
	 * @param functionName The config prefix from {@link HandleBotEvents#initClass(JTS3ServerMod_Interface, de.stefan1200.jts3serverquery.JTS3ServerQuery, String)}
	 * @param type Log error level
	 * @param msg Message as String for the bot log
	 * @param outputToSystemOut Set to <code>true</code>, if this should be printed to the system standard output stream. In most cases this should be <code>false</code> for plugins.
	 * @see JTS3ServerMod_Interface#ERROR_LEVEL_DEBUG
	 * @see JTS3ServerMod_Interface#ERROR_LEVEL_INFO
	 * @see JTS3ServerMod_Interface#ERROR_LEVEL_WARNING
	 * @see JTS3ServerMod_Interface#ERROR_LEVEL_ERROR
	 * @see JTS3ServerMod_Interface#ERROR_LEVEL_CRITICAL
	 */
	public abstract void addLogEntry(String functionName, byte type, String msg, boolean outputToSystemOut);

	/**
	 * Add a throwable stack trace to the bot log.
	 * @param functionName The config prefix from {@link HandleBotEvents#initClass(JTS3ServerMod_Interface, de.stefan1200.jts3serverquery.JTS3ServerQuery, String)}
	 * @param e Throwable for the bot log
	 * @param outputToSystemOut Set to <code>true</code>, if this should be printed to the system standard output stream. In most cases this should be <code>false</code> for plugins.
	 */
	public abstract void addLogEntry(String functionName, Throwable e, boolean outputToSystemOut);
	
	/**
	 * Returns the client database id of a client with the given unique id.
	 * @param uniqueID The unique id of the client
	 * @return The client database id of the given client or -1 if not found.
	 */
	public abstract int getClientDBID(String uniqueID);
	
	/**
	 * Returns a clientlist entry of a single client.
	 * @param clientID The client id as int
	 * @return Returns an HashMap object of a single client, if the client was found in the last received clientlist, <code>null</code> if not.
	 * @since 6.1.0
	 */
	public abstract HashMap<String, String> getClientListEntry(int clientID);
	
	/**
	 * Removes a clientlist entry of a single client. Use this to update the clientlist cache, if the JTS3ServerMod kicks a client from the TS3 server.<br>
	 * <br>
	 * <b>Notice:</b><br>Only the clientlist cache will be updated. To kick a client from the TS3 server, use the JTS3ServerQuery!
	 * @param clientID The client id as int
	 * @return Returns <code>true</code>, if the client was found and removed from the last received clientlist, <code>false</code> if not.
	 * @since 6.1.3
	 */
	public abstract boolean removeClientListEntry(int clientID);
	
	/**
	 * Set a new channel name in the channellist cache entry of a single channel. Use this to update the channellist cache, if the JTS3ServerMod renames a channel.<br>
	 * <br>
	 * <b>Notice:</b><br>Only the channellist cache will be updated. To rename a channel on the TS3 server, use the JTS3ServerQuery!
	 * @param channelID The channel id as int
	 * @param newName The new channel name as String
	 * @return Returns <code>true</code>, if the channel was found and the name was changed successfully, <code>false</code> if not.
	 * @since 6.1.3
	 */
	public abstract boolean setChannelName(int channelID, String newName);
	
	/**
	 * Removes a channellist entry of a single channel. Use this to update the channellist cache, if the JTS3ServerMod deletes a channel.<br>
	 * <br>
	 * <b>Notice:</b><br>Only the channellist cache will be updated. To delete a channel on the TS3 server, use the JTS3ServerQuery!
	 * @param channelID The channel id as int
	 * @return Returns <code>true</code>, if the channel was found and removed from the last received channellist, <code>false</code> if not.
	 * @since 6.1.3
	 */
	public abstract boolean removeChannelListEntry(int channelID);
	
	/**
	 * Check if the local Client Database Cache is enabled in the bot configuration. This can be used at loadConfig() already.
	 * @return Returns <code>true</code>, if the local Client Database Cache is enabled in the bot configuration, <code>false</code> if not.
	 */
	public abstract boolean isLocalClientDatabaseCacheEnabled();
	
}