package de.stefan1200.jts3servermod.examplefunctions;

import java.util.BitSet;
import java.util.HashMap;
import java.util.StringTokenizer;
import java.util.TimerTask;
import java.util.Vector;

import de.stefan1200.jts3servermod.BotConfigurationException;
import de.stefan1200.jts3servermod.FunctionExceptionLog;
import de.stefan1200.jts3servermod.interfaces.HandleBotEvents;
import de.stefan1200.jts3servermod.interfaces.HandleClientList;
import de.stefan1200.jts3servermod.interfaces.HandleTS3Events;
import de.stefan1200.jts3servermod.interfaces.JTS3ServerMod_Interface;
import de.stefan1200.jts3servermod.interfaces.LoadConfiguration;
import de.stefan1200.jts3serverquery.JTS3ServerQuery;
import de.stefan1200.jts3serverquery.TS3ServerQueryException;
import de.stefan1200.util.ArrangedPropertiesWriter;

public class AutoKickTimer
implements HandleBotEvents, LoadConfiguration, HandleClientList, HandleTS3Events
{
	private String configPrefix = "";
	private JTS3ServerMod_Interface modClass = null;
	private JTS3ServerQuery queryLib = null;
	private boolean pluginEnabled = false;
	
	private int kickTime = -1;
	private String messageFile = null;
	private String message = null;
	private String customMessage = null;
	private Vector<Integer> groupList = new Vector<Integer>();
	private boolean groupListIgnore = true;
	private TimerTask timerAutoKick;
	private boolean kickNow = false;
	private long timeLastKickNow = 0;
	private boolean firstInit = true;
	private HashMap<Integer, Long> clientConnectedTime = new HashMap<Integer, Long>();
	private FunctionExceptionLog fel = new FunctionExceptionLog();
	
	public static void main(String[] args)
	{
		// Unused dummy, but makes class selection at jar file export in Eclipse easier. :)
	}

	public void initClass(JTS3ServerMod_Interface modClass, JTS3ServerQuery queryLib, String prefix)
	{
		this.modClass = modClass;
		this.queryLib = queryLib;
		configPrefix = prefix.trim();
	}

	public void handleOnBotConnect()
	{
		if (!pluginEnabled) return;
		
		String msg = "";
		
		// Create a comma separated list of specified server groups.
		StringBuffer sGroupList = new StringBuffer();
		for (int groupID : groupList)
		{
			if (sGroupList.length() != 0)
			{
				sGroupList.append(", ");
			}
			
			sGroupList.append(groupID);
		}
		
		// Create a message to show what this function do on bot connect.
		if (sGroupList.length() == 0)
			msg = "After " + Integer.toString(kickTime) + " minutes online time all clients will be kicked!";
		else
			msg = "After " + Integer.toString(kickTime) + " minutes all clients " + (groupListIgnore ? "except members " : "") + "of the following server groups will be kicked: " + sGroupList.toString();
		
		// Print this message to the bot log and standard system output.
		modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_INFO, msg, true);
	}

	public void handleAfterCacheUpdate()
	{
		if (!pluginEnabled) return;
		
	}

	public void activate()
	{
		if (!pluginEnabled) return;
		
		// Stop a possible running TimerTask
		if (timerAutoKick != null) timerAutoKick.cancel();
		timerAutoKick = null;
		
		// Create a new TimerTask with the auto kick check interval of 60 seconds.
		timerAutoKick = new TimerTask()
		{
			public void run()
			{
				kickNow = true;
			}
		};
		modClass.addBotTimer(timerAutoKick, (1 * 60) * 1000, (1 * 60) * 1000);
		timeLastKickNow = System.currentTimeMillis();
	}

	public void disable()
	{
		if (!pluginEnabled) return;
		
		// Stop a possible running TimerTask
		if (timerAutoKick != null) timerAutoKick.cancel();
		timerAutoKick = null;
		kickNow = false;
	}

	public void unload()
	{
		// Doing some cleanup here
		customMessage = null;
		message = null;
		messageFile = null;
		clientConnectedTime = null;
		groupList = null;
	}

	public boolean multipleInstances()
	{
		// Don't allow multiple instances of this function per bot.
		return false;
	}

	public int getAPIBuild()
	{
		return 4;
	}
	
	public String getCopyright()
	{
		return "Auto Kick Timer function v1.2 (30.04.2016) created by Stefan \"Stefan1200\" Martens, [url]http://www.stefan1200.de[/url]";
	}

	public void initConfig(ArrangedPropertiesWriter config)
	{
		// The following three config keys will be always needed.
		config.addKey(configPrefix + "_time", "After how many minutes all clients should be kicked?", "5");
		config.addKey(configPrefix + "_group_list", "A comma separated list (without spaces) of server group ids.\nDepends on the given mode, this server groups can be ignored or only this server groups will be checked!\nIf no server groups should be ignored, set no server groups here and select the group list mode ignore!");
		config.addKey(configPrefix + "_group_list_mode", "Select one of the two modes for the server group list.\nignore = The selected server groups will be ignored.\nonly = Only the selected server groups will be checked.", "ignore");
		
		// The following config key is only needed, if the bot configuration is file based (and is not using MySQL).
		if (modClass.getMySQLConnection() == null) config.addKey(configPrefix + "_file", "Path to file which contains the Auto Kick Timer message.", "%apphome%config/server1/autokicktimermessages.cfg");
		
		// The following config key should not be saved to file, if the bot configuration is file based. But it will be saved into the MySQL database.
		config.addKey(configPrefix + "_message", "Auto Kick Timer message, the client get this message as kick message.\nYou can use the following keywords, which will be replaced:\n%AUTO_KICK_TIME% - This will be replaced with the auto kick time in minutes\nOnly 80 characters are possible for kick messages!", modClass.getMySQLConnection() != null);
	}

	public boolean loadConfig(ArrangedPropertiesWriter config, boolean slowMode)
	throws BotConfigurationException, NumberFormatException
	{
		String lastNumberValue = "";
		String temp = null;
		pluginEnabled = false;
		
		try
		{
			// Reading the group list from bot config, a comma separated list (without spaces) of server group ids was set by the user.
			// Will be saved into a Vector
			temp = null;
			groupList.clear();
			temp = config.getValue(configPrefix + "_group_list");
			lastNumberValue = configPrefix + "_group_list";
			if (temp != null && temp.length() > 0)
			{
				StringTokenizer st = new StringTokenizer(temp, ",", false);
				while (st.hasMoreTokens())
				{
					groupList.addElement(Integer.parseInt(st.nextToken().trim()));
				}
			}
	
			// Reading the group list mode from bot config. Two states are possible, a boolean is the target class member variable.
			groupListIgnore = !config.getValue(configPrefix + "_group_list_mode", "ignore").trim().equalsIgnoreCase("only");
			
			// Reading the time from bot config. Prepare for a possible NumberFormatException and saving the config key for a better error message.
			lastNumberValue = configPrefix + "_time";
			temp = config.getValue(configPrefix + "_time");
			if (temp == null) throw new NumberFormatException();
			kickTime = Integer.parseInt(temp.trim());
			
			// Just ensure, that the kick time is at least 1 minute.
			if (kickTime < 1)
			{
				kickTime = 1;
			}
			
			// Getting the file path and name of the AutoKick message file. Will be null in MySQL mode, because the value does not exists.
			messageFile = config.getValue(configPrefix + "_file");
			
			// Reading the AutoKick message into the ArrangedPropertiesWriter config instance using the JTS3ServerMod loadMessages method.
			// If the MySQL mode is active, loadMessages just return true. The message was already loaded from MySQL database by the ArrangedPropertiesWriter. 
			String[] configKeys = {configPrefix + "_message"};
			if (!modClass.loadMessages(configPrefix, "_file", configKeys))
			{
				throw new BotConfigurationException("Auto Kick Timer message could not be loaded!");
			}
			
			// Getting the AutoKick message from the ArrangedPropertiesWriter config instance.
			message = config.getValue(configKeys[0]);
			if (message == null || message.length() == 0)
			{
				throw new BotConfigurationException("Auto Kick Timer message missing in config!");
			}
			
			// Replace the own message variables using own method.
			createMessage();
			
			// If no exception was thrown until this line, just allow this plugin to be activated.
			pluginEnabled = true;
		}
		catch (NumberFormatException e)
		{
			// Catch the NumberFormatException and change the message to include the wrong config value while keeping the stack trace and throw the modified exception again.
			NumberFormatException nfe = new NumberFormatException("Config value of \"" + lastNumberValue + "\" is not a number! Current value: " + config.getValue(lastNumberValue, "not set"));
			nfe.setStackTrace(e.getStackTrace());
			throw nfe;
		}
		
		return pluginEnabled;
	}

	public void setListModes(BitSet listOptions)
	{
		// LIST_GROUPS is needed to know which server groups a client has. This allows this plugin to check only clients from specified server groups.
		listOptions.set(JTS3ServerMod_Interface.LIST_GROUPS);
		
		// LIST_TIMES is needed to know how long a client is already connected.
		listOptions.set(JTS3ServerMod_Interface.LIST_TIMES);
	}

	public void handleClientCheck(Vector<HashMap<String, String>> clientList)
	{
		if (!pluginEnabled) return;
		
		// Check connection time of all clients, who was already connected at bot start
		if (firstInit)
		{
			firstInit = false;
			
			HashMap<String, String> fullClientInfo;
			
			// Go through the whole client list
			for (HashMap<String, String> clientInfo : clientList)
			{
				// Check real Teamspeak 3 clients only, ignore query clients
				if (clientInfo.get("client_type").equals("0"))
				{
					// Parse the client id as int
					int clientID = Integer.parseInt(clientInfo.get("clid"));
					
					// Skip if we already have the connection time
					if (clientConnectedTime.containsKey(clientID))
						continue;
					
					try
					{
						// Request the full client info to get the connection_connected_time value
						fullClientInfo = queryLib.getInfo(JTS3ServerQuery.INFOMODE_CLIENTINFO, clientID);
						
						// Save connection time
						clientConnectedTime.put(clientID, (System.currentTimeMillis() - Long.parseLong(fullClientInfo.get("connection_connected_time"))));
					}
					catch (Exception e)
					{
						modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_INFO, "Error while requesting connection time from TS3 server for client \"" + clientInfo.get("client_nickname") + "\" (db id: " + clientInfo.get("client_database_id") + ")!", false);
						modClass.addLogEntry(configPrefix, e, false);
					}
				}
			}
		}
		
		// Start checking all clients after the TimerTask sets kickNow to true (only once per minute)
		if (kickNow)
		{
			kickNow = false;
			
			// Go through the whole client list
			for (HashMap<String, String> clientInfo : clientList)
			{
				// Check real Teamspeak 3 clients only, ignore query clients
				if (clientInfo.get("client_type").equals("0"))
				{
					// Parse the client id as int
					int clientID = Integer.parseInt(clientInfo.get("clid"));
					
					// Skip if we don't have the connection time
					if (!clientConnectedTime.containsKey(clientID))
						continue;
					
					// Check if the client is in the selected server group
					boolean result = modClass.isGroupListed(clientInfo.get("client_servergroups"), groupList);
					
					// Depends on the group list mode negate the result
					if ((groupListIgnore ? !result : result))
					{
						// Get the online time of the client
						long diff = System.currentTimeMillis() - clientConnectedTime.get(clientID);
						
						// Check if the client already exceeds the auto kick time
						if (diff > (kickTime * 60 * 1000))
						{
							// Kick the client if the client already exceeds the auto kick time
							kickClient(clientInfo, customMessage, clientID);
						}
					}
				}
			}
			
			timeLastKickNow = System.currentTimeMillis();
		}
	}

	private void kickClient(HashMap<String, String> clientInfo, String kickMSG, int clientID)
	{
		try
		{
			// Send kick command to the Teamspeak 3 server
			queryLib.kickClient(clientID, false, kickMSG);
			
			// Write successful kick action to the bot log
			modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_INFO, "Client \"" + clientInfo.get("client_nickname") + "\" (db id: " + clientInfo.get("client_database_id") + ") was online to long, client was kicked!", false);
			
			// Clear TS3ServerQueryException for this client, if there was one before, because kicking was successful now.
			fel.clearException(Integer.parseInt(clientInfo.get("client_database_id")));
		}
		catch (TS3ServerQueryException sqe)
		{
			// Checking if there is already a saved TS3ServerQueryException for this client.
			// This prevents flooding the log file with error messages, if a missing permission of the bot account on the Teamspeak 3 server is missing.
			if (!fel.existsException(sqe, Integer.parseInt(clientInfo.get("client_database_id"))))
			{
				// Since there is no saved TS3ServerQueryException for this client, save it now.
				fel.addException(sqe, Integer.parseInt(clientInfo.get("client_database_id")));
				
				// Adding error message to the bot log.
				modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_WARNING, "Client status of \"" + clientInfo.get("client_nickname") + "\" (db id: " + clientInfo.get("client_database_id") + ") is online to long, but an error occurred while kicking client!", false);
				if (sqe.getFailedPermissionID() < 0)
					modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_WARNING, sqe.toString(), false);
				else
					modClass.addLogEntry(configPrefix, sqe, false);
			}
		}
		catch (Exception e)
		{
			// Adding error message to the bot log.
			modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_ERROR, "Client status of \"" + clientInfo.get("client_nickname") + "\" (db id: " + clientInfo.get("client_database_id") + ") is online to long, but an error occurred while kicking client!", false);
			modClass.addLogEntry(configPrefix, e, false);
		}
	}
	
	private void createMessage()
	{
		// Create a copy of the String and replace the message variable.
		customMessage = new String(message);
		customMessage = customMessage.replace("%AUTO_KICK_TIME%", Integer.toString(kickTime));
		
		// Checking if the message length is valid or write a warning into the bot log.
		if (!modClass.isMessageLengthValid("kick", customMessage))
		{
			modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_WARNING, "Auto Kick Timer message is to long! Make sure that kick messages are not longer than " + Short.toString(modClass.getMaxMessageLength("kick")) + " characters (including spaces and BBCode)" + (modClass.getMySQLConnection() == null ? ", check file: "+messageFile : ""), true);
		}
	}

	public String[] botChatCommandList(HashMap<String, String> eventInfo, boolean isFullAdmin, boolean isAdmin)
	{
		if (!pluginEnabled) return null;
		
		// You can return multiple commands, if you just put more into the commands array.
		String[] commands = {"time"};
		return commands;
	}

	public String botChatCommandHelp(String command)
	{
		if (command.equalsIgnoreCase("time") || command.equalsIgnoreCase("timeleft"))
			return "Displays your current online time and how much time you have left.";
		
		// Always return null, if chat command is not handled by this function.
		return null;
	}

	public boolean handleChatCommands(String msg, HashMap<String, String> eventInfo, boolean isFullAdmin, boolean isAdmin)
	{
		if (!pluginEnabled) return false;
		
		if (msg.equalsIgnoreCase("time") || msg.equalsIgnoreCase("timeleft"))
		{
			String message = null;

			// Go through the whole client list, because I need the client_lastconnected value of the client
			Vector<HashMap<String, String>> clientList = modClass.getClientList();
			for (HashMap<String, String> clientInfo : clientList)
			{
				if (!clientInfo.get("clid").equals(eventInfo.get("invokerid")))
					continue;
				
				// Only check real Teamspeak 3 clients, ignore query clients
				if (clientInfo.get("client_type").equals("0"))
				{
					// Parse the client id as int
					int clientID = Integer.parseInt(eventInfo.get("invokerid"));
					
					// Check if the client is in the selected server group
					boolean result = modClass.isGroupListed(clientInfo.get("client_servergroups"), groupList);
					
					// Depends on the group list mode negate the result
					if ((groupListIgnore ? !result : result))
					{
						// Get the online time of the client
						long diff = System.currentTimeMillis() - clientConnectedTime.get(clientID);
						
						// Calculate the remaining time until kicking client
						long kickTimeFromNow = (kickTime * 60 * 1000) - diff;
						long newKickTime = 0;
						if (kickTimeFromNow < 0)
						{
							newKickTime = timeLastKickNow + 60000;
						}
						else
						{
							int minutes = (int)Math.ceil(kickTimeFromNow/60000.0);
							newKickTime = (timeLastKickNow + (minutes * 60000));
							long diff2 = (newKickTime - clientConnectedTime.get(clientID));
							if (diff2 < (kickTime * 60 * 1000))
								newKickTime += 60000;
						}
						
						message = "You are already connected since " + modClass.getDifferenceTime(clientConnectedTime.get(clientID), System.currentTimeMillis()) + 
								  ". You will be kicked in " + modClass.getDifferenceTime(System.currentTimeMillis(), newKickTime) + ".";
					}
					else
					{
						message = "You are already connected since " + modClass.getDifferenceTime(clientConnectedTime.get(clientID), System.currentTimeMillis()) + 
								  ". You will not be kicked, because you are not in the affected server group!";
					}
					
					try
					{
						// Sending above message to client.
						queryLib.sendTextMessage(clientID, JTS3ServerQuery.TEXTMESSAGE_TARGET_CLIENT, message);
						
						// Clear TS3ServerQueryException for this client, if there was one before, because kicking was successful now.
						fel.clearException(Integer.parseInt(clientInfo.get("client_database_id")));
					}
					catch (TS3ServerQueryException sqe)
					{
						// Checking if there is already a saved TS3ServerQueryException for this client.
						// This prevents flooding the log file with error messages, if a missing permission of the bot account on the Teamspeak 3 server is missing.
						if (!fel.existsException(sqe, Integer.parseInt(clientInfo.get("client_database_id"))))
						{
							// Since there is no saved TS3ServerQueryException for this client, save it now.
							fel.addException(sqe, Integer.parseInt(clientInfo.get("client_database_id")));
							
							// Adding error message to the bot log.
							modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_ERROR, "Cannot send chat command answer to client \"" + clientInfo.get("client_nickname") + "\" (db id: " + clientInfo.get("client_database_id") + "), an error occurred while sending message to client!", false);
							modClass.addLogEntry(configPrefix, sqe, false);
						}
					}
					
					break;
				}
			}
			
			return true;
		}
		
		return false;
	}

	public void handleTS3Events(String eventType, HashMap<String, String> eventInfo)
	{
		if (eventType.equalsIgnoreCase("notifycliententerview")) // Client connected to TS3 server
		{
			try
			{
				// Check real Teamspeak 3 clients only, ignore query clients
				if (eventInfo.get("client_type").equals("0"))
				{
					// Save connection time
					clientConnectedTime.put(Integer.parseInt(eventInfo.get("clid")), System.currentTimeMillis());
				}
			}
			catch (Exception e)
			{
			}
		}
		else if (eventType.equalsIgnoreCase("notifyclientleftview")) // Client disconnected from TS3 server
		{
			// Remove saved connection time
			clientConnectedTime.remove(Integer.parseInt(eventInfo.get("clid")));
		}
	}
}
