package de.stefan1200.jts3servermod;

import java.util.Vector;

import de.stefan1200.jts3serverquery.TS3ServerQueryException;

/**
 * The FunctionExceptionLog class can save exceptions of the type TS3ServerQueryException. If wanted, it can also save an id (like a client database id) to this exception.
 * The TS3ServerQueryException can occur very often until the problem was solved (maybe a missing permission of the bot on the Teamspeak 3 server).
 * If the exception occur again, you can check if this exception was already thrown (for this id) and prevent writing an error message to the log file again.<br><br>
 * 
 * <b>Example, fel is a class member variable:</b><br>
 * <pre> private FunctionExceptionLog <b>fel</b> = new FunctionExceptionLog();
 * <br> ************ more class content ***************<br>
 * try
 * {
 * 	queryLib.kickClient(clientID, false, kickMSG);
 * 	<b>fel</b>.clearException(Integer.parseInt(clientInfo.get("client_database_id")));
 * }
 * catch (TS3ServerQueryException sqe)
 * {
 * 	if (!<b>fel</b>.existsException(sqe, Integer.parseInt(clientInfo.get("client_database_id"))))
 * 	{
 * 		<b>fel</b>.addException(sqe, Integer.parseInt(clientInfo.get("client_database_id")));
 * 		modClass.addLogEntry(configPrefix, JTS3ServerMod_Interface.ERROR_LEVEL_ERROR, "An error occurred while kicking client \"" + clientInfo.get("client_nickname") + "\" (db id: " + clientInfo.get("client_database_id") + ")!", false);
 * 		modClass.addLogEntry(configPrefix, sqe, false);
 * 	}
 * }</pre><br>
 * The author of the JTS3ServerMod is not responsible for any damage or data loss!<br><br>
 * 
 * <b>E-Mail:</b><br><a href="mailto:info@stefan1200.de">info@stefan1200.de</a><br><br>
 * <b>Homepage:</b><br><a href="http://www.stefan1200.de" target="_blank">http://www.stefan1200.de</a>
 * @author Stefan Martens
 * @see TS3ServerQueryException
 *
 */
public class FunctionExceptionLog
{
	Vector<TS3ServerQueryException> exceptionLog = new Vector<TS3ServerQueryException>();
	Vector<Integer> exceptionTargetID = new Vector<Integer>();
	
	/**
	 * Add the TS3ServerQueryException to the list.
	 * @param exception The TS3ServerQueryException you want to add
	 */
	public void addException(TS3ServerQueryException exception)
	{
		addException(exception, -1);
	}
	
	/**
	 * Add the TS3ServerQueryException to the list, tied to an id (like a client database id).
	 * @param exception The TS3ServerQueryException you want to add.
	 * @param targetID An id (like a client database id) for the exception.
	 */
	public void addException(TS3ServerQueryException exception, int targetID)
	{
		exceptionLog.addElement(exception);
		exceptionTargetID.addElement(targetID);
	}
	
	/**
	 * Check if this exception was already saved. The error id and failed permission id of the TS3ServerQueryException will be compared.
	 * @param exception The TS3ServerQueryException
	 * @return <code>true</code>, if this exception was already saved, <code>false</code> if not.
	 */
	public boolean existsException(TS3ServerQueryException exception)
	{
		return existsException(exception, -1);
	}
	
	/**
	 * Check if this exception was already saved. The error id and failed permission id of the TS3ServerQueryException will be compared. The target id has to match also.
	 * @param exception The TS3ServerQueryException
	 * @param targetID An id (like a client database id) of the exception.
	 * @return <code>true</code>, if this exception was already saved, <code>false</code> if not.
	 */
	public boolean existsException(TS3ServerQueryException exception, int targetID)
	{
		for (int i=0; i<exceptionLog.size(); i++)
		{
			if (exception.getErrorID() == exceptionLog.elementAt(i).getErrorID() && exception.getFailedPermissionID() == exceptionLog.elementAt(i).getFailedPermissionID() && exceptionTargetID.elementAt(i) == targetID)
				return true;
		}
		
		return false;
	}
	
	/**
	 * Clears the TS3ServerQueryException of the given id (like a client database id).
	 * @param targetID An id (like a client database id) of the exception.
	 */
	public void clearException(int targetID)
	{
		int i = -1;
		while ((i = exceptionTargetID.indexOf(targetID)) >= 0)
		{
			exceptionLog.removeElementAt(i);
			exceptionTargetID.removeElementAt(i);
		}
	}
	
	/**
	 * Clears all saved TS3ServerQueryExceptions.
	 */
	public void clearAllExceptions()
	{
		exceptionLog.clear();
		exceptionTargetID.clear();
	}
}
